import tkinter as tk
from tkinter import ttk, filedialog, messagebox
import threading
import time
import os
from compressor import compress_folder
from decompressor import decompress_sor_file

class ITunesStyleApp(tk.Tk):
    def __init__(self):
        super().__init__()
        self.title('SOR Archiver')
        self.geometry('700x500')
        self.resizable(False, False)
        self.configure(bg='#d6d6d6')
        self.folder_path = ''
        self.sor_path = ''
        self.output_path = ''
        self.output_dir = ''
        self.compress_start_time = None
        self.decompress_start_time = None
        self._make_style()
        self._make_widgets()

    def _make_style(self):
        style = ttk.Style()
        style.theme_use('clam')
        style.configure('TButton', font=('Arial', 11), padding=6)
        style.configure('TLabel', font=('Arial', 11), background='#e6e6e6')
        style.configure('TFrame', background='#e6e6e6')
        style.configure('Sidebar.TFrame', background='#e0e0e0')
        style.configure('Sidebar.TLabel', background='#e0e0e0', font=('Arial', 12, 'bold'))
        style.configure('Title.TLabel', background='#b0b0b0', font=('Arial', 16, 'bold'))
        style.configure('Lime.Horizontal.TProgressbar',
                        troughcolor='#f0f0f0',
                        bordercolor='#b2ffb2',
                        background='#aaff77',
                        lightcolor='#eaffcc',
                        darkcolor='#77ff77',
                        thickness=16)

    def _make_widgets(self):
        # タイトルバー風
        title_bar = ttk.Frame(self, style='Title.TFrame')
        title_bar.pack(fill='x')
        ttk.Label(title_bar, text='SOR Archiver', style='Title.TLabel').pack(side='left', padx=12, pady=8)

        # メインエリア
        main_frame = ttk.Frame(self)
        main_frame.pack(fill='both', expand=True)

        # サイドバー
        sidebar = ttk.Frame(main_frame, width=140, style='Sidebar.TFrame')
        sidebar.pack(side='left', fill='y')
        ttk.Label(sidebar, text='ライブラリ', style='Sidebar.TLabel').pack(anchor='w', padx=12, pady=(16, 4))
        self.mode_var = tk.StringVar(value='compress')
        ttk.Radiobutton(sidebar, text='圧縮', variable=self.mode_var, value='compress', command=self._switch_mode).pack(anchor='w', padx=20, pady=2)
        ttk.Radiobutton(sidebar, text='解凍', variable=self.mode_var, value='decompress', command=self._switch_mode).pack(anchor='w', padx=20, pady=2)

        # メインパネル
        self.panel = ttk.Frame(main_frame)
        self.panel.pack(side='left', fill='both', expand=True, padx=8, pady=8)
        self._make_compress_panel()
        self._make_decompress_panel()
        self._switch_mode()

    def _make_compress_panel(self):
        self.compress_panel = ttk.Frame(self.panel)
        ttk.Label(self.compress_panel, text='圧縮するフォルダを選択', font=('Arial', 12)).pack(pady=8)
        ttk.Button(self.compress_panel, text='フォルダ選択', command=self.select_folder).pack(pady=2)
        self.folder_label = ttk.Label(self.compress_panel, text='未選択', font=('Arial', 10))
        self.folder_label.pack(pady=2)
        ttk.Button(self.compress_panel, text='圧縮開始', command=self.start_compress).pack(pady=8)
        self.compress_progress = ttk.Progressbar(self.compress_panel, style='Lime.Horizontal.TProgressbar', mode='determinate')
        self.compress_progress.pack(fill='x', padx=20, pady=8)
        self.compress_status = ttk.Label(self.compress_panel, text='準備完了')
        self.compress_status.pack(pady=2)

    def _make_decompress_panel(self):
        self.decompress_panel = ttk.Frame(self.panel)
        ttk.Label(self.decompress_panel, text='解凍するSORファイルを選択', font=('Arial', 12)).pack(pady=8)
        ttk.Button(self.decompress_panel, text='SORファイル選択', command=self.select_sor_file).pack(pady=2)
        self.sor_label = ttk.Label(self.decompress_panel, text='未選択', font=('Arial', 10))
        self.sor_label.pack(pady=2)
        ttk.Button(self.decompress_panel, text='解凍開始', command=self.start_extract).pack(pady=8)
        self.decompress_progress = ttk.Progressbar(self.decompress_panel, style='Lime.Horizontal.TProgressbar', mode='determinate')
        self.decompress_progress.pack(fill='x', padx=20, pady=8)
        self.decompress_status = ttk.Label(self.decompress_panel, text='準備完了')
        self.decompress_status.pack(pady=2)

    def _switch_mode(self):
        for widget in self.panel.winfo_children():
            widget.pack_forget()
        if self.mode_var.get() == 'compress':
            self.compress_panel.pack(fill='both', expand=True)
        else:
            self.decompress_panel.pack(fill='both', expand=True)

    def select_folder(self):
        path = filedialog.askdirectory(title="圧縮するフォルダを選択")
        if path:
            self.folder_path = path
            self.folder_label.config(text=os.path.basename(path))

    def select_sor_file(self):
        path = filedialog.askopenfilename(
            title="SORファイルを選択",
            filetypes=[("SOR Archive", "*.sor"), ("All Files", "*.*")]
        )
        if path:
            self.sor_path = path
            self.sor_label.config(text=os.path.basename(path))

    def start_compress(self):
        if not self.folder_path:
            messagebox.showerror('エラー', '圧縮するフォルダを選択してください')
            return
        output_file = filedialog.asksaveasfilename(
            title="SORアーカイブを保存",
            defaultextension=".sor",
            filetypes=[("SOR Archive", "*.sor"), ("All Files", "*.*")]
        )
        if not output_file:
            return
        self.output_path = output_file
        self.compress_progress['value'] = 0
        self.compress_status.config(text="圧縮中...")
        threading.Thread(target=self._compress_thread, daemon=True).start()

    def start_extract(self):
        if not self.sor_path:
            messagebox.showerror('エラー', '.sorファイルを選択してください')
            return
        output_dir = filedialog.askdirectory(title="出力先フォルダを選択")
        if not output_dir:
            return
        self.output_dir = output_dir
        self.decompress_progress['value'] = 0
        self.decompress_status.config(text="解凍中...")
        threading.Thread(target=self._extract_thread, daemon=True).start()

    def update_compress_progress(self, progress, status):
        self.compress_progress['value'] = progress
        self.compress_status.config(text=status)
        self.update()

    def update_decompress_progress(self, progress, status):
        self.decompress_progress['value'] = progress
        self.decompress_status.config(text=status)
        self.update()

    def _compress_thread(self):
        try:
            success = compress_folder(
                self.folder_path,
                self.output_path,
                self.update_compress_progress
            )
            if success:
                self.after(0, lambda: self.compress_status.config(text="完了"))
                self.after(0, lambda: messagebox.showinfo('完了', '圧縮が完了しました'))
            else:
                self.after(0, lambda: self.compress_status.config(text="失敗"))
                self.after(0, lambda: messagebox.showerror('エラー', '圧縮に失敗しました'))
        except Exception as e:
            self.after(0, lambda: self.compress_status.config(text="エラー"))
            self.after(0, lambda: messagebox.showerror('エラー', f'圧縮エラー: {e}'))
        finally:
            self.after(0, lambda: self.compress_progress.config(value=0))

    def _extract_thread(self):
        try:
            success = decompress_sor_file(
                self.sor_path,
                self.output_dir,
                self.update_decompress_progress
            )
            if success:
                self.after(0, lambda: self.decompress_status.config(text="完了"))
                self.after(0, lambda: messagebox.showinfo('完了', '解凍が完了しました'))
            else:
                self.after(0, lambda: self.decompress_status.config(text="失敗"))
                self.after(0, lambda: messagebox.showerror('エラー', '解凍に失敗しました'))
        except Exception as e:
            self.after(0, lambda: self.decompress_status.config(text="エラー"))
            self.after(0, lambda: messagebox.showerror('エラー', f'解凍エラー: {e}'))
        finally:
            self.after(0, lambda: self.decompress_progress.config(value=0))

if __name__ == '__main__':
    app = ITunesStyleApp()
    app.mainloop()