import os
import struct
from bwt import bwt_decode
from mtf import mtf_decode
from huffman import huffman_decode

def decompress_file(compressed_data: bytes, huffman_tree: bytes, original_size: int) -> bytes:
    try:
        if not compressed_data or not huffman_tree or original_size == 0:
            return b""
        if len(huffman_tree) < 4:
            return b""
        bwt_index = struct.unpack('<I', huffman_tree[:4])[0]
        tree_data = huffman_tree[4:]
        mtf_data = huffman_decode(compressed_data, tree_data)
        bwt_data = mtf_decode(mtf_data)
        original_data = bwt_decode(bwt_data, bwt_index)
        return original_data
    except Exception:
        return b""

def decompress_sor_file(sor_path: str, output_dir: str, progress_callback=None) -> bool:
    try:
        if not os.path.exists(sor_path):
            return False
        os.makedirs(output_dir, exist_ok=True)
        with open(sor_path, 'rb') as f:
            magic = f.read(4)
            if magic != b'SOR\x00':
                return False
            file_count_bytes = f.read(4)
            if len(file_count_bytes) < 4:
                return False
            file_count = struct.unpack('<I', file_count_bytes)[0]
            if file_count == 0:
                return True
            for i in range(file_count):
                try:
                    path_len_bytes = f.read(2)
                    if len(path_len_bytes) < 2:
                        return False
                    path_len = struct.unpack('<H', path_len_bytes)[0]
                    path_bytes = f.read(path_len)
                    if len(path_bytes) < path_len:
                        return False
                    file_path = path_bytes.decode('utf-8')
                    size_bytes = f.read(8)
                    if len(size_bytes) < 8:
                        return False
                    original_size = struct.unpack('<Q', size_bytes)[0]
                    compressed_size_bytes = f.read(4)
                    if len(compressed_size_bytes) < 4:
                        return False
                    compressed_size = struct.unpack('<I', compressed_size_bytes)[0]
                    tree_size_bytes = f.read(4)
                    if len(tree_size_bytes) < 4:
                        return False
                    tree_size = struct.unpack('<I', tree_size_bytes)[0]
                    compressed_data = f.read(compressed_size)
                    if len(compressed_data) < compressed_size:
                        return False
                    huffman_tree = f.read(tree_size)
                    if len(huffman_tree) < tree_size:
                        return False
                    original_data = decompress_file(compressed_data, huffman_tree, original_size)
                    if original_data:
                        output_path = os.path.join(output_dir, file_path)
                        output_dir_path = os.path.dirname(output_path)
                        if output_dir_path:
                            os.makedirs(output_dir_path, exist_ok=True)
                        with open(output_path, 'wb') as out_file:
                            out_file.write(original_data)
                        if progress_callback:
                            progress = (i + 1) / file_count * 100
                            progress_callback(progress, f"Decompressing: {file_path}")
                except Exception:
                    continue
        return True
    except Exception:
        return False