import os
import struct
from bwt import bwt_encode
from mtf import mtf_encode
from huffman import huffman_encode

def compress_file(file_path: str):
    with open(file_path, 'rb') as f:
        data = f.read()
    original_size = len(data)
    if original_size == 0:
        return b"", b"", 0
    bwt_data, bwt_index = bwt_encode(data)
    mtf_data = mtf_encode(bwt_data)
    huff_data, huff_tree = huffman_encode(mtf_data)
    index_bytes = struct.pack('<I', bwt_index)
    huff_tree = index_bytes + huff_tree
    return huff_data, huff_tree, original_size

def compress_folder(folder_path: str, output_path: str, progress_callback=None) -> bool:
    try:
        files_to_compress = []
        for root, dirs, files in os.walk(folder_path):
            for file in files:
                file_path = os.path.join(root, file)
                rel_path = os.path.relpath(file_path, folder_path)
                files_to_compress.append((file_path, rel_path))
        if not files_to_compress:
            return False

        compressed_files = []
        total_files = len(files_to_compress)
        for i, (file_path, rel_path) in enumerate(files_to_compress):
            try:
                compressed_data, huffman_tree, original_size = compress_file(file_path)
                if compressed_data and huffman_tree:
                    compressed_files.append({
                        'path': rel_path,
                        'data': compressed_data,
                        'tree': huffman_tree,
                        'size': original_size
                    })
                if progress_callback:
                    progress = (i + 1) / total_files * 100
                    progress_callback(progress, f"Compressing: {rel_path}")
            except Exception:
                continue

        if not compressed_files:
            return False

        with open(output_path, 'wb') as f:
            f.write(b'SOR\x00')
            f.write(struct.pack('<I', len(compressed_files)))
            for file_info in compressed_files:
                path_bytes = file_info['path'].encode('utf-8')
                f.write(struct.pack('<H', len(path_bytes)))
                f.write(path_bytes)
                f.write(struct.pack('<Q', file_info['size']))
                f.write(struct.pack('<I', len(file_info['data'])))
                f.write(struct.pack('<I', len(file_info['tree'])))
                f.write(file_info['data'])
                f.write(file_info['tree'])
        return True
    except Exception:
        return False