<?php
define('BASE_DIR', __DIR__);
define('MAX_FILE_SIZE', 1024 * 1024);

$path = isset($_GET['path']) ? $_GET['path'] : '';
$path = str_replace(['../', '..\\'], '', $path);
$fullPath = BASE_DIR . ($path ? '/' . $path : '');

if (!file_exists($fullPath)) {
    $fullPath = BASE_DIR;
    $path = '';
}

$isFile = is_file($fullPath);
$isDir = is_dir($fullPath);

function formatSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    return round($bytes, 2) . ' ' . $units[$pow];
}

function getDirectoryContents($dir) {
    $items = scandir($dir);
    $directories = [];
    $files = [];
    
    foreach ($items as $item) {
        if ($item === '.' || $item === '..') continue;
        
        $itemPath = $dir . '/' . $item;
        $relativePath = str_replace(BASE_DIR . '/', '', $itemPath);
        
        $info = [
            'name' => $item,
            'path' => $relativePath,
            'size' => filesize($itemPath),
            'modified' => filemtime($itemPath),
        ];
        
        if (is_dir($itemPath)) {
            $directories[] = $info;
        } else {
            $files[] = $info;
        }
    }
    
    usort($directories, function($a, $b) {
        return strcasecmp($a['name'], $b['name']);
    });
    usort($files, function($a, $b) {
        return strcasecmp($a['name'], $b['name']);
    });
    
    return array_merge($directories, $files);
}

function getFileContent($filePath) {
    if (filesize($filePath) > MAX_FILE_SIZE) {
        return false;
    }
    
    $content = file_get_contents($filePath);
    
    if (preg_match('/[\x00-\x08\x0B\x0C\x0E-\x1F]/', substr($content, 0, 1024))) {
        return false;
    }
    
    return $content;
}

function getBreadcrumbs($path) {
    $parts = $path ? explode('/', $path) : [];
    $breadcrumbs = [];
    $currentPath = '';
    
    if (empty($parts)) {
        return $breadcrumbs;
    }
    
    foreach ($parts as $part) {
        $currentPath .= ($currentPath ? '/' : '') . $part;
        $breadcrumbs[] = ['name' => $part, 'path' => $currentPath];
    }
    
    return $breadcrumbs;
}

function isImageFile($filename) {
    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    return in_array($ext, ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'svg', 'ico', 'tiff', 'tif']);
}

function isTextFile($filename) {
    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    $textExts = ['txt', 'md', 'log', 'ini', 'cfg', 'conf', 'yml', 'yaml', 'toml', 
                 'php', 'html', 'htm', 'css', 'js', 'json', 'xml', 'sql', 'sh', 'bat',
                 'py', 'java', 'cpp', 'c', 'h', 'cs', 'rb', 'go', 'rs', 'swift', 'kt',
                 'pl', 'r', 'lua', 'vim', 'gitignore', 'htaccess', 'env'];
    return in_array($ext, $textExts);
}

function isArchiveOrBinary($filename) {
    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    $binaryExts = ['zip', 'rar', 'tar', 'gz', '7z', 'lzh', 'sor', 'fro', 'bz2', 'xz', 'arj', 'cab', 'lz', 'lz4', 'zst',
                   'exe', 'dll', 'so', 'dylib', 'bin', 'dat', 'iso', 'dmg', 'img', 'msi', 'deb', 'rpm',
                   'pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'odt', 'ods', 'odp'];
    return in_array($ext, $binaryExts);
}

function isReadmeOrLicense($filename) {
    $name = strtolower(basename($filename));
    $patterns = ['readme', 'license', 'licence', 'copying', 'authors', 'contributors', 'changelog', 'history'];
    foreach ($patterns as $pattern) {
        if (strpos($name, $pattern) === 0) {
            return true;
        }
    }
    return false;
}

function getFileType($filename) {
    if (is_dir($filename)) return 'Directory';
    $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    
    $types = [
        'php' => 'PHP Script',
        'html' => 'HTML Document',
        'htm' => 'HTML Document',
        'css' => 'CSS Stylesheet',
        'js' => 'JavaScript',
        'json' => 'JSON Data',
        'xml' => 'XML Document',
        'sql' => 'SQL Script',
        'txt' => 'Text File',
        'md' => 'Markdown',
        'log' => 'Log File',
        'ini' => 'Configuration',
        'cfg' => 'Configuration',
        'conf' => 'Configuration',
        'yml' => 'YAML Config',
        'yaml' => 'YAML Config',
        'py' => 'Python Script',
        'java' => 'Java Source',
        'cpp' => 'C++ Source',
        'c' => 'C Source',
        'cs' => 'C# Source',
        'rb' => 'Ruby Script',
        'go' => 'Go Source',
        'rs' => 'Rust Source',
        'jpg' => 'JPEG Image',
        'jpeg' => 'JPEG Image',
        'png' => 'PNG Image',
        'gif' => 'GIF Image',
        'bmp' => 'Bitmap Image',
        'svg' => 'SVG Image',
        'webp' => 'WebP Image',
        'zip' => 'ZIP Archive',
        'rar' => 'RAR Archive',
        'tar' => 'TAR Archive',
        'gz' => 'GZIP Archive',
        '7z' => '7-Zip Archive',
        'lzh' => 'LZH Archive',
        'sor' => 'SOR Archive',
        'fro' => 'FRO Archive',
    ];
    
    return isset($types[$ext]) ? $types[$ext] : strtoupper($ext) . ' File';
}

function generateFolderIcon() {
    header('Content-Type: image/png');
    $img = imagecreatetruecolor(16, 16);
    imagesavealpha($img, true);
    $transparent = imagecolorallocatealpha($img, 0, 0, 0, 127);
    imagefill($img, 0, 0, $transparent);
    
    $folderBody = imagecolorallocate($img, 255, 223, 110);
    $folderDark = imagecolorallocate($img, 230, 190, 80);
    $folderTab = imagecolorallocate($img, 255, 200, 90);
    $border = imagecolorallocate($img, 180, 140, 50);
    
    imagefilledrectangle($img, 2, 6, 14, 14, $folderBody);
    imagefilledrectangle($img, 2, 4, 6, 6, $folderTab);
    imagefilledrectangle($img, 3, 7, 14, 9, $folderDark);
    
    imagerectangle($img, 2, 6, 14, 14, $border);
    imageline($img, 2, 4, 6, 4, $border);
    imageline($img, 6, 4, 6, 6, $border);
    imageline($img, 2, 4, 2, 6, $border);
    
    imagepng($img);
    imagedestroy($img);
    exit;
}

function generateFileIcon($ext = '') {
    header('Content-Type: image/png');
    $img = imagecreatetruecolor(16, 16);
    imagesavealpha($img, true);
    $transparent = imagecolorallocatealpha($img, 0, 0, 0, 127);
    imagefill($img, 0, 0, $transparent);
    
    $colorMap = [
        'php' => [155, 89, 182],
        'html' => [231, 76, 60],
        'htm' => [231, 76, 60],
        'css' => [52, 152, 219],
        'js' => [241, 196, 15],
        'json' => [241, 196, 15],
        'txt' => [236, 240, 241],
        'md' => [149, 165, 166],
        'xml' => [230, 126, 34],
        'sql' => [46, 204, 113],
        'py' => [52, 73, 94],
        'java' => [211, 84, 0],
        'cpp' => [41, 128, 185],
        'c' => [44, 62, 80],
        'cs' => [108, 92, 231],
        'rb' => [192, 57, 43],
        'go' => [0, 173, 216],
        'rs' => [206, 92, 0],
        'swift' => [240, 81, 56],
        'kt' => [139, 96, 201],
        'log' => [149, 165, 166],
        'ini' => [189, 195, 199],
        'cfg' => [189, 195, 199],
        'conf' => [189, 195, 199],
        'yml' => [203, 67, 53],
        'yaml' => [203, 67, 53],
        'sh' => [76, 175, 80],
        'bat' => [76, 175, 80],
    ];
    
    if (isset($colorMap[$ext])) {
        $fileColor = imagecolorallocate($img, $colorMap[$ext][0], $colorMap[$ext][1], $colorMap[$ext][2]);
        $borderColor = imagecolorallocate($img, max(0, $colorMap[$ext][0]-40), max(0, $colorMap[$ext][1]-40), max(0, $colorMap[$ext][2]-40));
    } else {
        $fileColor = imagecolorallocate($img, 236, 240, 241);
        $borderColor = imagecolorallocate($img, 189, 195, 199);
    }
    
    $white = imagecolorallocate($img, 255, 255, 255);
    
    imagefilledpolygon($img, [4, 2, 4, 14, 13, 14, 13, 5, 10, 2], 5, $fileColor);
    imagefilledpolygon($img, [10, 2, 13, 5, 10, 5], 3, $white);
    
    imagepolygon($img, [4, 2, 4, 14, 13, 14, 13, 5, 10, 2], 5, $borderColor);
    imageline($img, 10, 2, 10, 5, $borderColor);
    imageline($img, 10, 5, 13, 5, $borderColor);
    
    if (isset($colorMap[$ext])) {
        $lines = imagecolorallocate($img, 255, 255, 255);
        imageline($img, 6, 6, 11, 6, $lines);
        imageline($img, 6, 8, 11, 8, $lines);
        imageline($img, 6, 10, 11, 10, $lines);
        imageline($img, 6, 12, 9, 12, $lines);
    }
    
    imagepng($img);
    imagedestroy($img);
    exit;
}

function generateReadmeIcon() {
    header('Content-Type: image/png');
    $img = imagecreatetruecolor(16, 16);
    imagesavealpha($img, true);
    $transparent = imagecolorallocatealpha($img, 0, 0, 0, 127);
    imagefill($img, 0, 0, $transparent);
    
    $bookColor = imagecolorallocate($img, 52, 152, 219);
    $bookDark = imagecolorallocate($img, 41, 128, 185);
    $white = imagecolorallocate($img, 255, 255, 255);
    
    imagefilledrectangle($img, 3, 3, 13, 13, $bookColor);
    imagerectangle($img, 3, 3, 13, 13, $bookDark);
    imageline($img, 8, 3, 8, 13, $bookDark);
    
    imageline($img, 5, 6, 6, 6, $white);
    imageline($img, 5, 8, 6, 8, $white);
    imageline($img, 5, 10, 6, 10, $white);
    
    imageline($img, 10, 6, 11, 6, $white);
    imageline($img, 10, 8, 11, 8, $white);
    imageline($img, 10, 10, 11, 10, $white);
    
    imagepng($img);
    imagedestroy($img);
    exit;
}

function generateLicenseIcon() {
    header('Content-Type: image/png');
    $img = imagecreatetruecolor(16, 16);
    imagesavealpha($img, true);
    $transparent = imagecolorallocatealpha($img, 0, 0, 0, 127);
    imagefill($img, 0, 0, $transparent);
    
    $scaleColor = imagecolorallocate($img, 149, 165, 166);
    $scaleDark = imagecolorallocate($img, 127, 140, 141);
    
    imagefilledrectangle($img, 7, 2, 9, 4, $scaleColor);
    
    imagefilledpolygon($img, [3, 5, 6, 5, 5, 10], 3, $scaleColor);
    imagefilledpolygon($img, [10, 5, 13, 5, 12, 10], 3, $scaleColor);
    
    imagefilledrectangle($img, 7, 4, 9, 14, $scaleColor);
    imagefilledrectangle($img, 5, 13, 11, 14, $scaleColor);
    
    imagerectangle($img, 7, 2, 9, 4, $scaleDark);
    imageline($img, 3, 5, 6, 5, $scaleDark);
    imageline($img, 10, 5, 13, 5, $scaleDark);
    imagerectangle($img, 5, 13, 11, 14, $scaleDark);
    
    imagepng($img);
    imagedestroy($img);
    exit;
}

if (isset($_GET['icon'])) {
    if ($_GET['icon'] === 'folder') {
        generateFolderIcon();
    } elseif ($_GET['icon'] === 'readme') {
        generateReadmeIcon();
    } elseif ($_GET['icon'] === 'license') {
        generateLicenseIcon();
    } else {
        generateFileIcon($_GET['icon']);
    }
}

if (isset($_GET['download']) && $_GET['download']) {
    $downloadPath = str_replace(['../', '..\\'], '', $_GET['download']);
    $downloadFullPath = BASE_DIR . '/' . $downloadPath;
    
    if (file_exists($downloadFullPath) && is_file($downloadFullPath)) {
        header('Content-Description: File Transfer');
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . basename($downloadFullPath) . '"');
        header('Content-Length: ' . filesize($downloadFullPath));
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        readfile($downloadFullPath);
        exit;
    }
}

if (isset($_GET['copy']) && $_GET['copy']) {
    $copyPath = str_replace(['../', '..\\'], '', $_GET['copy']);
    $copyFullPath = BASE_DIR . '/' . $copyPath;
    
    if (file_exists($copyFullPath) && is_file($copyFullPath)) {
        $content = file_get_contents($copyFullPath);
        $copySuccess = true;
    }
}

$breadcrumbs = getBreadcrumbs($path);
?>
<!DOCTYPE html>
<html>
<head>
<meta charset="Shift_JIS">
<title>ReV1ew - <?php echo htmlspecialchars($path ?: ''); ?></title>
</head>
<body bgcolor="#F0F0F0" text="#000000" link="#0066CC" vlink="#0066CC" alink="#FF9900">

<table width="100%" border="0" cellpadding="0" cellspacing="0">
<tr>
<td bgcolor="#007ACC" style="background: linear-gradient(to bottom, #007ACC 0%, #005A9E 100%);">
<table width="100%" border="0" cellpadding="12" cellspacing="0">
<tr>
<td>
<font face="Segoe UI, Tahoma, Arial" size="6" color="#FFFFFF"><b>ReV1ew</b></font>
</td>
</tr>
</table>
</td>
</tr>
<tr>
<td bgcolor="#E6E6E6" style="border-bottom: 1px solid #CCCCCC;">
<table width="100%" border="0" cellpadding="8" cellspacing="0">
<tr>
<td>
<font face="Segoe UI, Tahoma, Arial" size="2">
<a href="?" style="text-decoration: none;">Home</a>
<?php if (!empty($breadcrumbs)): ?>
<?php foreach ($breadcrumbs as $i => $crumb): ?>
<font color="#666666"> &gt; </font>
<?php if ($i === count($breadcrumbs) - 1): ?>
<font color="#000000"><b><?php echo htmlspecialchars($crumb['name']); ?></b></font>
<?php else: ?>
<a href="?path=<?php echo urlencode($crumb['path']); ?>"><?php echo htmlspecialchars($crumb['name']); ?></a>
<?php endif; ?>
<?php endforeach; ?>
<?php endif; ?>
</font>
</td>
</tr>
</table>
</td>
</tr>
</table>

<br>

<?php if ($isDir): ?>
<table width="98%" border="0" cellpadding="0" cellspacing="0" align="center" bgcolor="#FFFFFF" style="border: 1px solid #CCCCCC;">
<tr bgcolor="#F5F5F5" style="background: linear-gradient(to bottom, #FAFAFA 0%, #EFEFEF 100%); border-bottom: 1px solid #CCCCCC;">
<td width="30" style="border-right: 1px solid #E0E0E0;"><font face="Segoe UI, Tahoma, Arial" size="2">&nbsp;</font></td>
<td style="border-right: 1px solid #E0E0E0; padding: 8px;"><font face="Segoe UI, Tahoma, Arial" size="2" color="#333333"><b>Name</b></font></td>
<td width="100" style="border-right: 1px solid #E0E0E0; padding: 8px;" align="right"><font face="Segoe UI, Tahoma, Arial" size="2" color="#333333"><b>Size</b></font></td>
<td width="180" style="padding: 8px;"><font face="Segoe UI, Tahoma, Arial" size="2" color="#333333"><b>Date Modified</b></font></td>
</tr>
<?php
$items = getDirectoryContents($fullPath);
$rowColor = true;
foreach ($items as $item):
    $isItemDir = is_dir(BASE_DIR . '/' . $item['path']);
    $ext = strtolower(pathinfo($item['name'], PATHINFO_EXTENSION));
    $bgColor = $rowColor ? '#FFFFFF' : '#F9F9F9';
    $rowColor = !$rowColor;
?>
<tr bgcolor="<?php echo $bgColor; ?>" style="border-bottom: 1px solid #F0F0F0;" onmouseover="this.style.backgroundColor='#E5F3FF';" onmouseout="this.style.backgroundColor='<?php echo $bgColor; ?>';">
<td align="center" style="border-right: 1px solid #F0F0F0; padding: 6px;">
<?php if ($isItemDir): ?>
<img src="?icon=folder" width="16" height="16" alt="DIR">
<?php else: ?>
<img src="?icon=<?php echo htmlspecialchars($ext); ?>" width="16" height="16" alt="FILE">
<?php endif; ?>
</td>
<td style="border-right: 1px solid #F0F0F0; padding: 6px;">
<font face="Segoe UI, Tahoma, Arial" size="2">
<a href="?path=<?php echo urlencode($item['path']); ?>" style="text-decoration: none;">
<?php echo htmlspecialchars($item['name']); ?>
</a>
</font>
</td>
<td align="right" style="border-right: 1px solid #F0F0F0; padding: 6px;">
<font face="Segoe UI, Tahoma, Arial" size="2" color="#666666">
<?php echo $isItemDir ? '' : formatSize($item['size']); ?>
</font>
</td>
<td style="padding: 6px;">
<font face="Segoe UI, Tahoma, Arial" size="2" color="#666666">
<?php echo date('Y/m/d H:i', $item['modified']); ?>
</font>
</td>
</tr>
<?php endforeach; ?>
</table>

<?php
$readmeFile = null;
$licenseFile = null;
foreach ($items as $item) {
    if (!is_dir(BASE_DIR . '/' . $item['path'])) {
        if (isReadmeOrLicense($item['name']) && !$licenseFile && !$readmeFile) {
            $name = strtolower($item['name']);
            if (strpos($name, 'readme') === 0) {
                $readmeFile = $item;
            } elseif (strpos($name, 'license') === 0 || strpos($name, 'licence') === 0) {
                $licenseFile = $item;
            }
        }
    }
}

if ($readmeFile):
    $readmePath = BASE_DIR . '/' . $readmeFile['path'];
    $readmeContent = getFileContent($readmePath);
    if ($readmeContent !== false && isTextFile($readmePath)):
?>
<br>
<table width="98%" border="0" cellpadding="0" cellspacing="0" align="center" bgcolor="#FFFFFF" style="border: 1px solid #CCCCCC;">
<tr bgcolor="#F5F5F5" style="background: linear-gradient(to bottom, #FAFAFA 0%, #EFEFEF 100%); border-bottom: 1px solid #CCCCCC;">
<td style="padding: 10px;">
<img src="?icon=readme" width="16" height="16" alt="README" style="vertical-align: middle; margin-right: 5px;">
<font face="Segoe UI, Tahoma, Arial" size="3" color="#000000">
<b><?php echo htmlspecialchars($readmeFile['name']); ?></b>
</font>
</td>
</tr>
<tr>
<td style="padding: 15px;">
<pre style="margin: 0; padding: 0; white-space: pre-wrap; word-wrap: break-word;"><font face="Consolas, Courier New, monospace" size="2" color="#000000"><?php echo htmlspecialchars($readmeContent); ?></font></pre>
</td>
</tr>
</table>
<?php 
    endif;
endif;

if ($licenseFile):
    $licensePath = BASE_DIR . '/' . $licenseFile['path'];
    $licenseContent = getFileContent($licensePath);
    if ($licenseContent !== false && isTextFile($licensePath)):
?>
<br>
<table width="98%" border="0" cellpadding="0" cellspacing="0" align="center" bgcolor="#FFFFFF" style="border: 1px solid #CCCCCC;">
<tr bgcolor="#F5F5F5" style="background: linear-gradient(to bottom, #FAFAFA 0%, #EFEFEF 100%); border-bottom: 1px solid #CCCCCC;">
<td style="padding: 10px;">
<img src="?icon=license" width="16" height="16" alt="LICENSE" style="vertical-align: middle; margin-right: 5px;">
<font face="Segoe UI, Tahoma, Arial" size="3" color="#000000">
<b><?php echo htmlspecialchars($licenseFile['name']); ?></b>
</font>
</td>
</tr>
<tr>
<td style="padding: 15px;">
<pre style="margin: 0; padding: 0; white-space: pre-wrap; word-wrap: break-word;"><font face="Consolas, Courier New, monospace" size="2" color="#000000"><?php echo htmlspecialchars($licenseContent); ?></font></pre>
</td>
</tr>
</table>
<?php 
    endif;
endif;
?>

<?php elseif ($isFile): ?>
<?php if ($path): ?>
<table width="98%" border="0" cellpadding="8" cellspacing="0" align="center">
<tr>
<td>
<a href="?path=<?php echo urlencode(dirname($path) === '.' ? '' : dirname($path)); ?>" style="text-decoration: none;">
<font face="Segoe UI, Tahoma, Arial" size="2" color="#0066CC"><b>&lt; Back to folder</b></font>
</a>
</td>
</tr>
</table>
<?php endif; ?>

<table width="98%" border="0" cellpadding="0" cellspacing="0" align="center" bgcolor="#FFFFFF" style="border: 1px solid #CCCCCC;">
<tr bgcolor="#F5F5F5" style="background: linear-gradient(to bottom, #FAFAFA 0%, #EFEFEF 100%); border-bottom: 1px solid #CCCCCC;">
<td style="padding: 10px;">
<font face="Segoe UI, Tahoma, Arial" size="3" color="#000000">
<b><?php echo htmlspecialchars(basename($fullPath)); ?></b>
</font>
<font face="Segoe UI, Tahoma, Arial" size="2" color="#666666">
&nbsp;-&nbsp;<?php echo formatSize(filesize($fullPath)); ?>
</font>
</td>
<td align="right" style="padding: 10px;">
<a href="?download=<?php echo urlencode($path); ?>" style="text-decoration: none; background-color: #007ACC; color: #FFFFFF; padding: 6px 12px; border-radius: 3px; margin-right: 5px;">
<font face="Segoe UI, Tahoma, Arial" size="2"><b>Download</b></font>
</a>
<?php if (isTextFile($fullPath) && filesize($fullPath) < MAX_FILE_SIZE): ?>
<a href="?copy=<?php echo urlencode($path); ?>&path=<?php echo urlencode($path); ?>" style="text-decoration: none; background-color: #68217A; color: #FFFFFF; padding: 6px 12px; border-radius: 3px;">
<font face="Segoe UI, Tahoma, Arial" size="2"><b>Copy</b></font>
</a>
<?php endif; ?>
</td>
</tr>
</table>

<?php if (isset($copySuccess) && $copySuccess): ?>
<br>
<table width="98%" border="0" cellpadding="10" cellspacing="0" align="center" bgcolor="#DFF0D8" style="border: 1px solid #D6E9C6;">
<tr>
<td>
<font face="Segoe UI, Tahoma, Arial" size="2" color="#3C763D">
<b>Content copied to clipboard!</b> You can paste it anywhere.
</font>
</td>
</tr>
</table>
<textarea id="copyarea" style="position: absolute; left: -9999px;"><?php echo htmlspecialchars($content); ?></textarea>
<script>
var copyarea = document.getElementById('copyarea');
copyarea.select();
document.execCommand('copy');
</script>
<?php endif; ?>

<br>

<table width="98%" border="0" cellpadding="0" cellspacing="0" align="center" bgcolor="#F9F9F9" style="border: 1px solid #DDDDDD;">
<tr>
<td style="padding: 10px;">
<table width="100%" border="0" cellpadding="3" cellspacing="0">
<tr>
<td width="120"><font face="Segoe UI, Tahoma, Arial" size="2" color="#666666"><b>File Type:</b></font></td>
<td><font face="Segoe UI, Tahoma, Arial" size="2" color="#333333"><?php echo getFileType($fullPath); ?></font></td>
</tr>
<tr>
<td><font face="Segoe UI, Tahoma, Arial" size="2" color="#666666"><b>Size:</b></font></td>
<td><font face="Segoe UI, Tahoma, Arial" size="2" color="#333333"><?php echo formatSize(filesize($fullPath)); ?> (<?php echo number_format(filesize($fullPath)); ?> bytes)</font></td>
</tr>
<tr>
<td><font face="Segoe UI, Tahoma, Arial" size="2" color="#666666"><b>Modified:</b></font></td>
<td><font face="Segoe UI, Tahoma, Arial" size="2" color="#333333"><?php echo date('Y/m/d H:i:s', filemtime($fullPath)); ?></font></td>
</tr>
</table>
</td>
</tr>
</table>

<br>

<?php if (isImageFile($fullPath)): ?>
<table width="98%" border="0" cellpadding="20" cellspacing="0" align="center" bgcolor="#FFFFFF" style="border: 1px solid #CCCCCC;">
<tr>
<td align="center">
<img src="data:image/<?php echo pathinfo($fullPath, PATHINFO_EXTENSION); ?>;base64,<?php echo base64_encode(file_get_contents($fullPath)); ?>" alt="<?php echo htmlspecialchars(basename($fullPath)); ?>" style="max-width:100%; border: 1px solid #DDDDDD;">
</td>
</tr>
</table>
<?php elseif (isTextFile($fullPath) && !isArchiveOrBinary($fullPath)): ?>
<?php
$content = getFileContent($fullPath);
if ($content !== false):
?>
<table width="98%" border="0" cellpadding="12" cellspacing="0" align="center" bgcolor="#FFFFFF" style="border: 1px solid #CCCCCC;">
<tr>
<td>
<pre style="margin: 0; padding: 0; white-space: pre-wrap; word-wrap: break-word;"><font face="Consolas, Courier New, monospace" size="2" color="#000000"><?php echo htmlspecialchars($content); ?></font></pre>
</td>
</tr>
</table>
<?php else: ?>
<table width="98%" border="0" cellpadding="30" cellspacing="0" align="center" bgcolor="#FFFFFF" style="border: 1px solid #CCCCCC;">
<tr>
<td align="center">
<font face="Segoe UI, Tahoma, Arial" size="3" color="#999999">
File is too large to display
</font>
</td>
</tr>
</table>
<?php endif; ?>
<?php else: ?>
<table width="98%" border="0" cellpadding="30" cellspacing="0" align="center" bgcolor="#FFFFFF" style="border: 1px solid #CCCCCC;">
<tr>
<td align="center">
<font face="Segoe UI, Tahoma, Arial" size="3" color="#999999">
Binary or archive file - preview not available
</font>
</td>
</tr>
</table>
<?php endif; ?>

<?php endif; ?>

<br><br>

<table width="100%" border="0" cellpadding="10" cellspacing="0">
<tr>
<td bgcolor="#E6E6E6" align="center" style="border-top: 1px solid #CCCCCC;">
<font face="Segoe UI, Tahoma, Arial" size="1" color="#666666">
ReV1ew File Browser
</font>
</td>
</tr>
</table>

</body>
</html>